Imports System
Imports iNTERFACEWARE.IGC

Module IGCexampleMain
    Dim WithEvents Processor As iNTERFACEWARE.IGC.MessageProcessor = New iNTERFACEWARE.IGC.MessageProcessor()
    Dim StopEvent As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)

    Sub Main(ByVal Args() As String)
        'Usage matches default executable parameters passed in by Iguana: $hostname$ $port$ $channel$ $component$
        Dim Usage As String = "Usage: IGCexample.exe Hostname Port Channel Component"

        Try
            If Args.Length <> 4 Then
                Throw New System.Exception("Invalid number of command-line arguments.")
            End If

            Dim Hostname As String = Args(0)
            Dim Port As Integer = System.Convert.ToInt16(Args(1))
            Dim Channel As String = Args(2)
            Dim Component As String = Args(3)

            If Component <> "destination" Then
                Throw New System.Exception("Invalid component - only 'destination' is supported by this plugin.")
            End If

            System.Console.WriteLine("Connecting to Iguana")
            Processor.connect(Hostname, Port, Channel, Component)

            ' Wait for the stop event, which is generated when the Iguana channel is stopped
            StopEvent.WaitOne()

            Return
        Catch ex As iNTERFACEWARE.IGC.Exception
            System.Console.WriteLine(ex.Description)
        Catch ex As System.Exception
            System.Console.WriteLine(ex.Message)
        End Try

        System.Console.WriteLine(Usage)
    End Sub

    ' This event is triggered when a message is received.
    Private Sub Processor_OnMessageReceived(ByVal sender As Object, ByVal e As System.EventArgs, _
        ByVal Message As String, ByVal MessageResponse As MessageReceivedResponse) _
        Handles Processor.OnMessageReceived
        ' Exceptions cannot be thrown out of this handler. Any exceptions thrown will abort this program.
        ' Logging and exception handling must be performed in this handler.
        Try
            System.Console.WriteLine("*** Message Received ***")
            ' Message processing goes here. This example just prints the received message as is.
            System.Console.WriteLine(Message)
            System.Console.WriteLine("")
            ' Tell Iguana that the message has been successfully processed.
            MessageResponse.MessageProcessed = True

            ' If the plugin is unable to process the message right away (for example,
            ' because the plugin is attempting to communicate with a database that is
            ' sometimes unavailable), you can tell Iguana to wait a specified number of
            ' milliseconds before resending the message.
            '
            '    MessageResponse.MessageProcessed = false
            '    MessageResponse.ResendMessage = true
            '    MessageResponse.ResendMessageDelay = 1000

            ' Or you can tell Iguana to stop the channel and resend the message after the
            ' channel is restarted.
            '
            '    MessageResponse.MessageProcessed = false
            '    MessageResponse.ResendMessage = true
            '    MessageResponse.StopChannel = true

            ' You can use StopChannel, StopChannelErrorFlag, and StopChannelReason to stop the Iguana 
            ' channel with or without an error flag, and provide a reason for the stop request.
            '
            '    MessageResponse.StopChannel = True
            '    MessageResponse.StopChannelErrorFlag = True
            '    MessageResponse.StopChannelReason = "An error occurred."

            ' You can send additional error/info/debug log messages to Iguana with addLogMessage().
            '
            '    MessageResponse.addLogMessage(IGClogMessageType.Error, "Error log message")
            '    MessageResponse.addLogMessage(IGClogMessageType.Info, "Info log message")
            '    MessageResponse.addLogMessage(IGClogMessageType.Debug, "Debug log message")
            '    MessageResponse.addLogMessage(IGClogMessageType.Warning, "Warning log message")

            ' You can use addLogValue() to send additional Key/Value pair logging information to Iguana.
            '
            '    MessageResponse.AddLogValue("MyField", "Message successfully processed.")
        Catch ex As iNTERFACEWARE.IGC.Exception
            System.Console.WriteLine(ex.Description)
        Catch ex As System.Exception
            System.Console.WriteLine(ex.Message)
        End Try
    End Sub

    ' This event is triggered when the Iguana channel is shut down.
    Private Sub Processor_OnDisconnect(ByVal sender As Object, ByVal e As System.EventArgs, _
        ByVal ReasonFlags As MessageProcessor.IGCdisconnectReasonFlags, ByVal ReasonDescription As String) _
        Handles Processor.OnDisconnect
        ' Exceptions cannot be thrown out of this handler. Any exceptions thrown will abort this program.
        ' Logging and exception handling must be performed in this handler.
        Try
            If (MessageProcessor.isDisconnectByError(ReasonFlags)) Then
                System.Console.WriteLine("Connection closed by error: " + ReasonDescription)
            ElseIf (MessageProcessor.isDisconnectByPlugin(ReasonFlags)) Then
                System.Console.WriteLine("Plugin closed the connection")
            ElseIf (MessageProcessor.isDisconnectByIguana(ReasonFlags)) Then
                System.Console.WriteLine("Iguana closed the connection")
            Else
                System.Console.WriteLine("Connection closed: " + ReasonDescription)
            End If
            StopEvent.Set()
        Catch ex As iNTERFACEWARE.IGC.Exception
            System.Console.WriteLine(ex.Description)
        Catch ex As System.Exception
            System.Console.WriteLine(ex.Message)
        End Try
    End Sub

End Module

